<?php
/**
 * Plugin Name:       SlipMgr Affiliate Network
 * Description:       Monetize your boating content. Display a marina search engine and earn commissions on slip bookings.
 * Version:           3.4.4
 * Author:            Karl Kiebel
 * License:           GPL-2.0-or-later
 * Text Domain:       slipmgr-affiliate
 */

if ( ! defined( 'ABSPATH' ) ) exit;

define( 'SLIPMGR_AFFILIATE_VERSION', '3.4.4' );
// Point this to your actual Access Manager installation
define( 'SLIPMGR_NETWORK_API_URL', 'https://slipmgr.com/wp-json/access-manager/v1' );

// --- Settings ---
function slipmgr_affiliate_register_settings() {
    // FIX: Added 'sanitize_text_field' callback
    register_setting('slipmgr_affiliate_options', 'slipmgr_affiliate_key', 'sanitize_text_field');
    add_settings_section('slipmgr_affiliate_main', 'Affiliate Settings', null, 'slipmgr-affiliate-settings');
    add_settings_field('slipmgr_affiliate_key', 'Your Affiliate API Key', 'slipmgr_affiliate_key_cb', 'slipmgr-affiliate-settings', 'slipmgr_affiliate_main');
}
add_action('admin_init', 'slipmgr_affiliate_register_settings');

function slipmgr_affiliate_key_cb() {
    $val = get_option('slipmgr_affiliate_key');
    echo "<input type='text' name='slipmgr_affiliate_key' value='" . esc_attr($val) . "' class='regular-text' />";
    echo "<p class='description'>Enter the API Key provided by the SlipMgr Network to track your commissions.</p>";
}

function slipmgr_affiliate_add_admin_menu() {
    add_menu_page(
        'SlipMgr Affiliate', 
        'SlipMgr Affiliate', 
        'manage_options', 
        'slipmgr-affiliate', 
        'slipmgr_affiliate_render_settings_page',
        'dashicons-networking',
        30
    );
}
add_action('admin_menu', 'slipmgr_affiliate_add_admin_menu');

function slipmgr_affiliate_render_settings_page() {
    ?>
    <div class="wrap">
        <h1>SlipMgr Affiliate Settings</h1>
        <form method="post" action="options.php">
            <?php
            settings_fields('slipmgr_affiliate_options');
            do_settings_sections('slipmgr-affiliate-settings');
            submit_button();
            ?>
        </form>
        
        <hr style="margin: 30px 0;">
        
        <h2>How to Use</h2>
        <p>Add the marina search engine to any WordPress Page, Post, or Widget by pasting the shortcode below.</p>

        <div style="background: #fff; border: 1px solid #ccd0d4; padding: 20px; max-width: 800px;">
            <h3 style="margin-top:0;">Basic Usage</h3>
            <code>[slipmgr_affiliate_search]</code>
            <p class="description">This will display the default map centered on the US with a standard blue theme.</p>

            <h3 style="margin-top:25px;">Configuration Attributes</h3>
            <p>Customize the look and feel to match your website using these options:</p>
            
            <table class="widefat striped">
                <thead>
                    <tr>
                        <th style="font-weight:bold;">Attribute</th>
                        <th style="font-weight:bold;">Description</th>
                        <th style="font-weight:bold;">Example</th>
                    </tr>
                </thead>
                <tbody>
                    <tr>
                        <td><code>height</code></td>
                        <td>Sets the height of the map container. Supports px, vh, or %.</td>
                        <td><code>height="800px"</code></td>
                    </tr>
                    <tr>
                        <td><code>color</code></td>
                        <td>Hex code for your brand color. Affects buttons and markers.</td>
                        <td><code>color="#E1306C"</code></td>
                    </tr>
                    <tr>
                        <td><code>lat</code></td>
                        <td>Latitude to center the map on load.</td>
                        <td><code>lat="25.7617"</code></td>
                    </tr>
                    <tr>
                        <td><code>long</code></td>
                        <td>Longitude to center the map on load.</td>
                        <td><code>long="-80.1918"</code></td>
                    </tr>
                    <tr>
                        <td><code>zoom</code></td>
                        <td>Initial zoom level (1-18). 10 is roughly city-wide view.</td>
                        <td><code>zoom="10"</code></td>
                    </tr>
                </tbody>
            </table>

            <h3 style="margin-top:25px;">Copy-Paste Examples</h3>
            
            <p><strong>1. Branded Region Map (e.g., Lake Tahoe in Green)</strong></p>
            <code style="display:block; padding:10px; background:#f0f0f1;">[slipmgr_affiliate_search height="700px" color="#2ecc71" lat="39.0968" long="-120.0324" zoom="10"]</code>

            <p><strong>2. Compact Sidebar Widget</strong></p>
            <code style="display:block; padding:10px; background:#f0f0f1;">[slipmgr_affiliate_search height="400px" zoom="8"]</code>
        </div>

        <div style="margin-top: 20px; padding-top: 20px; border-top: 1px solid #ccc; text-align: center; color: #666;">
            Powered by <a href="https://slipmgr.com" target="_blank" rel="noopener noreferrer" style="text-decoration: none; color: #0073aa;">SlipMgr.com</a>
        </div>
    </div>
    <?php
}

// --- Shortcode & Assets ---
function slipmgr_affiliate_enqueue_assets($attributes = []) {
    $manifest_path = plugin_dir_path( __FILE__ ) . 'dist/manifest.json';
    if ( ! file_exists( $manifest_path ) ) { return; }
    $manifest = json_decode( file_get_contents( $manifest_path ), true );
    
    $entry = $manifest['index.tsx'] ?? null;
    if (!$entry) return;

    // Robust file loading with cache busting based on file modification time
    $js_rel_path = 'dist/' . $entry['file'];
    $js_abs_path = plugin_dir_path( __FILE__ ) . $js_rel_path;
    $js_url = plugins_url( $js_rel_path, __FILE__ );
    $js_ver = file_exists($js_abs_path) ? filemtime($js_abs_path) : SLIPMGR_AFFILIATE_VERSION;
    
    // Removed 'wp-element' dependency as we bundle our own React to prevent conflicts
    wp_enqueue_script('slipmgr-affiliate-script', $js_url, [], $js_ver, true);
    
    wp_localize_script('slipmgr-affiliate-script', 'slipmgrAffiliateData', [
        'networkApiUrl' => SLIPMGR_NETWORK_API_URL,
        'affiliateKey' => get_option('slipmgr_affiliate_key', ''),
        'defaultCity' => $attributes['city'] ?? '',
        'defaultCategory' => $attributes['category'] ?? '',
        'height' => $attributes['height'] ?? '600px',
        'brandColor' => $attributes['color'] ?? '#0ea5e9', // Default Blue
        'defaultLat' => $attributes['lat'] ?? '',
        'defaultLong' => $attributes['long'] ?? '',
        'defaultZoom' => $attributes['zoom'] ?? ''
    ]);

    if (isset($entry['css'])) {
        foreach ($entry['css'] as $css) {
            $css_rel_path = 'dist/' . $css;
            $css_abs_path = plugin_dir_path( __FILE__ ) . $css_rel_path;
            $css_url = plugins_url($css_rel_path, __FILE__);
            $css_ver = file_exists($css_abs_path) ? filemtime($css_abs_path) : SLIPMGR_AFFILIATE_VERSION;
            
            wp_enqueue_style('slipmgr-affiliate-style', $css_url, [], $css_ver);
        }
    }
}

function slipmgr_affiliate_render_app($atts) {
    $attributes = shortcode_atts([
        'city' => '',
        'category' => '',
        'height' => '600px',
        'color' => '#0ea5e9',
        'lat' => '',
        'long' => '',
        'zoom' => ''
    ], $atts);
    
    slipmgr_affiliate_enqueue_assets($attributes);
    
    ob_start();
    ?>
    <div id="slipmgr-affiliate-root" style="min-height: <?php echo esc_attr($attributes['height']); ?>; background: #f3f4f6; border-radius: 8px; display: flex; align-items: center; justify-content: center;">Loading Marina Search...</div>
    <?php
    return ob_get_clean();
}
add_shortcode('slipmgr_affiliate_search', 'slipmgr_affiliate_render_app');

// Module type tag for Vite
add_filter('script_loader_tag', function($tag, $handle, $src) {
    if ('slipmgr-affiliate-script' === $handle) {
        // phpcs:ignore WordPress.WP.EnqueuedResources.NonEnqueuedScript
        return '<script type="module" src="' . esc_url($src) . '"></script>';
    }
    return $tag;
}, 10, 3);